#ifndef LAMPE_H_
#define LAMPE_H_

#include "repere.h"
#include "couleur.h"


/***** Classe permettant de grer l'attenuation constante, linaire et quadratique relative 
		un spectre de puissance lumineuse *****/

class Attenuation
{
	float lin, cons, quad;

	public:
		Attenuation(const float&, const float&, const float&);
		float operator ()(const float&) const;
};

Attenuation::Attenuation(const float& _l, const float& _c, const float& _q) : lin(_l), cons(_c), quad(_q)
{}

float Attenuation::operator ()(const float& distsq) const
{
	return min(1.0f, 1.0f/(lin*distsq + cons*sqrtf(distsq)));
}

// classe abstraire 
// Permet de driver en lampe ponctuelle, spot, ...

class Lampe
{ 
	public :
		Repere rPropreInv;		// repre propre inverse de la lampe
		SpectrePuiss Pa1m;		// Puissance lumineuse mise

		 Lampe(const Repere& pos=Repere(), const SpectrePuiss& sp=SpectrePuiss()) throw (RepereDegenere) :  rPropreInv(pos.Inverse()), Pa1m(sp) {}
         virtual SpectrePuiss Recu(const pR3 &p) const = 0; // Puissance recue en p (exprim en global) !                         
};

class LampeIso : public Lampe
{
	protected:
		Attenuation att;		// Attnuation (constante, linaire, quadratique)

	public:
		LampeIso(const Repere&, const SpectrePuiss&, const Attenuation& _a=Attenuation(0.00005f, 0.0f, 0.0f)) throw (RepereDegenere);
		SpectrePuiss Recu(const pR3 &p) const;
};

LampeIso::LampeIso(const Repere& pos, const SpectrePuiss& spec, const Attenuation& _a) throw (RepereDegenere): Lampe(pos, spec), att(_a)
{}


// Calcul de la puissance reue au point p, en tenant compte
// de l'attnuation

SpectrePuiss LampeIso::Recu(const pR3 &p) const
{
	vR3 P = rPropreInv*(p);
	return (Pa1m * att.operator()(P%P));
}


#endif // LAMPE_H_
