#ifndef ALGEBRE_H_
#define ALGEBRE_H_

#include <math.h>
#include "define.h"

class pR2
{
	public:

	float x,y;

	pR2(const float& a=0,const float& b=0);
	pR2(const pR2& p);

	pR2&  operator =(const pR2& p);
	pR2   operator -  () const;
	bool  operator == (const pR2 &p) const;
	bool  operator != (const pR2 &p) const;
	bool  operator < (const pR2 &p) const;
	bool  operator <= (const pR2 &p) const;
	bool  operator > (const pR2 &p) const;
	bool  operator >= (const pR2 &p) const;
};


pR2::pR2(const float& a, const float& b) : x(a),y(b) {}

pR2::pR2(const pR2& p):x(p.x),y(p.y) {}

pR2 pR2::operator - () const
{
	return pR2(-x,-y);
}

pR2& pR2::operator = (const pR2& p)
{
	x=p.x;
	y=p.y;
	return (*this);
}


class vR3
 {    public : float x,y,z;

  vR3 (float a=0, float b=0, float c=0) : x(a), y(b), z(c) {}
//                                     ____________________
// syntaxe C++ :
//
//  les champs sont OBLIGATOIREMENT construits avant 
//  le lancement du corps du constructeur {}. Cette syntaxe optionnelle
//  permet de spcifier leur construction. En l'absence de cette spcification,
//  le(s) champ(s) concern(s) est(sont construit(s) par dfaut

 };

class pR3
 {    public : float x,y,z;

  pR3 (float a=0, float b=0, float c=0) : x(a), y(b), z(c) {}

  pR3 (const vR3 &v) : x(v.x), y(v.y), z(v.z) {}

  operator vR3 () const { return vR3(x,y,z); }
//_______ 1 _____ _ 2 _
// syntaxe C++ :
//          1   oprateur de "cast" d'un pR3 en vR3 appel 
//                 * implicitement ds que ncessaire
//                 * explicitement si demand
//          2   mthode "const" suppose ne pas pouvoir modifier l'objet lanceur
//                 demande explicite au compilateur de vrifier l'absence d'instructions
//                 susceptibles de le modifier
 };

  inline vR3 operator * (float      a, const vR3 &v) {return vR3(  a*v.x,
                                                                   a*v.y,
                                                                   a*v.z); }
  inline pR3 operator * (float      a, const pR3 &p) {return pR3(  a*p.x,
                                                                   a*p.y,
                                                                   a*p.z); }
  inline vR3 operator * (const vR3 & v, float     a) {return vR3(  a*v.x,
                                                                   a*v.y,
                                                                   a*v.z); }
  inline pR3 operator * (const pR3 & p, float     a) {return pR3(  a*p.x,
                                                                   a*p.y,
                                                                   a*p.z); }

  inline vR3 operator / (const vR3 & v, float     a) {return vR3(  v.x/a,
                                                                   v.y/a,
                                                                   v.z/a); }
  inline pR3 operator / (const pR3 & p, float     a) {return pR3(  p.x/a,
                                                                   p.y/a,
                                                                   p.z/a); }

  inline vR3 operator + (const vR3 &u, const vR3 &v) {return vR3(u.x+v.x,
                                                                 u.y+v.y,
                                                                 u.z+v.z); }
  inline vR3 operator - (const vR3 &u, const vR3 &v) {return vR3(u.x-v.x,
                                                                 u.y-v.y,
                                                                 u.z-v.z); }

  inline vR3 operator - (const pR3 &P, const pR3 &p) {return vR3(P.x-p.x,
                                                                 P.y-p.y,
                                                                 P.z-p.z); }

  inline pR3 operator + (const pR3 &p, const vR3 &v) {return pR3(p.x+v.x,
                                                                 p.y+v.y,
                                                                 p.z+v.z); }
  inline pR3 operator - (const pR3 &p, const vR3 &v) {return pR3(p.x-v.x,
                                                                 p.y-v.y,
                                                                 p.z-v.z); }
  inline pR3 operator + (const vR3 &v, const pR3 &p) {return pR3(v.x+p.x,
                                                                 v.y+p.y,
                                                                 v.z+p.z); }
  inline pR3 operator - (const vR3 &v, const pR3 &p) {return pR3(v.x-p.x,
                                                                 v.y-p.y,
                                                                 v.z-p.z); }

  inline vR3 operator - (const vR3 & v)              {return vR3(-v.x,
                                                                 -v.y,
                                                                 -v.z); }
  inline pR3 operator - (const pR3 & v)              {return pR3(-v.x,
                                                                 -v.y,
                                                                 -v.z); }

  inline vR3 operator ^ (const vR3 &u, const vR3 &v) {return vR3(u.y*v.z-u.z*v.y,
                                                                 u.z*v.x-u.x*v.z,
                                                                 u.x*v.y-u.y*v.x); }

  inline float operator%(const vR3 &u, const vR3 &v) {return u.x*v.x
                                                            +u.y*v.y
                                                            +u.z*v.z; }

  inline float ProdMixte(const vR3 &u, const vR3 &v, const vR3 &w)
                                                     {return w.x*(u.y*v.z-u.z*v.y)
                                                            +w.y*(u.z*v.x-u.x*v.z)
                                                            +w.z*(u.x*v.y-u.y*v.x); }
  inline float Norme   (const pR3 &p) { return sqrt(p%p);  }
  inline pR3   Unitaire(const pR3 &p) { return p/Norme(p); }
  inline float Norme   (const vR3 &v) { return sqrt(v%v);  }
  inline vR3   Unitaire(const vR3 &v) { return v/Norme(v); }

class Transform
{
	public:

	vR3 vx,vy,vz,vt;

	Transform(double a =1.,double b =0.,double c =0.,
	         double d =0.,double e =1.,double f =0.,
				double g =0.,double h =0.,double i =1.,
				double t1=0.,double t2=0.,double t3=0.);
	Transform(const vR3 &x,const vR3 &y,const vR3 &z,const vR3 &t);

	Transform   operator +  (const Transform& m) const;
	Transform   operator -  (const Transform& m) const;
	Transform   operator *  (const double r) const;
	vR3         operator *  (const vR3& v) const;
	pR3         operator *  (const pR3& p) const;
	Transform   operator *  (const Transform& m) const;
	Transform   operator /  (const double r) const;

	double    det() const; // determinant
	Transform inv() const; // inverse
};

class CRep
{
	public:

	Transform mat, invMat;

	CRep();
	CRep(const vR3 &v1,const vR3 &v2,const vR3 &v3,const vR3 &v4);
	CRep(const Transform &A);

	CRep inv () const;

	virtual CRep   operator *  (const CRep &Tr) const;

	virtual vR3 Apply    (const vR3 &v) const;
	virtual vR3 invApply (const vR3 &v) const;
	virtual pR3 Apply    (const pR3 &p) const;
	virtual pR3 invApply (const pR3 &p) const;
};

//
// Methodes de la classe Transform
//

Transform::Transform(double  a,double  b,double  c,
                   double  d,double  e,double  f,
						 double  g,double  h,double  i,
						 double t1,double t2,double t3):vx(a,b,c),vy(d,e,f),vz(g,h,i),vt(t1,t2,t3){}

Transform::Transform(const vR3 &X,const vR3 &Y,const vR3 &Z,const vR3 &T):vx(X),vy(Y),vz(Z),vt(T){}

Transform Transform::operator + (const Transform &m) const
{
	return Transform(vx+m.vx,vy+m.vy,vz+m.vz,vt+m.vt);
}


Transform Transform::operator - (const Transform &m) const
{
	return Transform(vx-m.vx,vy-m.vy,vz-m.vz,vt-m.vt);
}

Transform Transform::operator * (const double r) const
{
	return Transform(vx*r,vy*r,vz*r,vt*r);
}


vR3 Transform::operator * (const vR3 &u) const
{
	return vR3(u.x*vx.x + u.y*vy.x + u.z*vz.x,
	           u.x*vx.y + u.y*vy.y + u.z*vz.y,
				  u.x*vx.z + u.y*vy.z + u.z*vz.z);
}

pR3 Transform::operator * (const pR3 &p) const
{
	return pR3(p.x*vx.x + p.y*vy.x + p.z*vz.x + vt.x,
	           p.x*vx.y + p.y*vy.y + p.z*vz.y + vt.y,
				  p.x*vx.z + p.y*vy.z + p.z*vz.z + vt.z);
}

Transform Transform::operator * (const Transform &m) const
{
	return Transform(vx.x*m.vx.x + vy.x*m.vx.y + vz.x*m.vx.z,
	                vx.y*m.vx.x + vy.y*m.vx.y + vz.y*m.vx.z,
						 vx.z*m.vx.x + vy.z*m.vx.y + vz.z*m.vx.z,
						 vx.x*m.vy.x + vy.x*m.vy.y + vz.x*m.vy.z,
						 vx.y*m.vy.x + vy.y*m.vy.y + vz.y*m.vy.z,
						 vx.z*m.vy.x + vy.z*m.vy.y + vz.z*m.vy.z,
						 vx.x*m.vz.x + vy.x*m.vz.y + vz.x*m.vz.z,
						 vx.y*m.vz.x + vy.y*m.vz.y + vz.y*m.vz.z,
						 vx.z*m.vz.x + vy.z*m.vz.y + vz.z*m.vz.z,
						 vx.x*m.vt.x + vy.x*m.vt.y + vz.x*m.vt.z + vt.x,
						 vx.y*m.vt.x + vy.y*m.vt.y + vz.y*m.vt.z + vt.y,
						 vx.z*m.vt.x + vy.z*m.vt.y + vz.z*m.vt.z + vt.z);
}

Transform Transform::operator / (const double r) const
{
	return Transform(vx/r,vy/r,vz/r,vt/r);
}

double Transform::det() const
{
	double det_1, pos, neg, temp;

	pos = neg = 0.0f;
	temp =  vx.x * vy.y * vz.z; if (temp >= 0.0f) pos += temp;  else  neg += temp;
	temp =  vx.y * vy.z * vz.x; if (temp >= 0.0f) pos += temp;  else  neg += temp;
	temp =  vx.z * vy.x * vz.y; if (temp >= 0.0f) pos += temp;  else  neg += temp;
	temp = -vx.z * vy.y * vz.x; if (temp >= 0.0f) pos += temp;  else  neg += temp;
	temp = -vx.y * vy.x * vz.z; if (temp >= 0.0f) pos += temp;  else  neg += temp;
	temp = -vx.x * vy.z * vz.y; if (temp >= 0.0f) pos += temp;  else  neg += temp;
	det_1 = pos + neg;

	return det_1;
}

Transform Transform::inv() const
{
	double det_1, pos, neg, temp;
	Transform Inv;

	//  * Calculate the determinant of submatrix A
	pos = neg = 0.0f;
	temp =  vx.x * vy.y * vz.z; if (temp >= 0.0f) pos += temp;  else  neg += temp;
	temp =  vx.y * vy.z * vz.x; if (temp >= 0.0f) pos += temp;  else  neg += temp;
	temp =  vx.z * vy.x * vz.y; if (temp >= 0.0f) pos += temp;  else  neg += temp;
	temp = -vx.z * vy.y * vz.x; if (temp >= 0.0f) pos += temp;  else  neg += temp;
	temp = -vx.y * vy.x * vz.z; if (temp >= 0.0f) pos += temp;  else  neg += temp;
	temp = -vx.x * vy.z * vz.y; if (temp >= 0.0f) pos += temp;  else  neg += temp;
	det_1 = pos + neg;

	// Is Singular ?
	if ((det_1 == 0.0f) || (fabs(det_1 / (pos - neg)) < 1e-15))
		throw "Transform Transform::inv() const - Singular matrix";

	//  Calculate inverse(A) = adj(A) / det(A)
	det_1 = 1.0f / det_1;
	Inv.vx.x =( (vy.y * vz.z - vy.z * vz.y) * det_1);
	Inv.vy.x =(-(vy.x * vz.z - vy.z * vz.x) * det_1);
	Inv.vz.x =( (vy.x * vz.y - vy.y * vz.x) * det_1);
	Inv.vx.y =(-(vx.y * vz.z - vx.z * vz.y) * det_1);
	Inv.vy.y =( (vx.x * vz.z - vx.z * vz.x) * det_1);
	Inv.vz.y =(-(vx.x * vz.y - vx.y * vz.x) * det_1);
	Inv.vx.z =( (vx.y * vy.z - vx.z * vy.y) * det_1);
	Inv.vy.z =(-(vx.x * vy.z - vx.z * vy.x) * det_1);
	Inv.vz.z =( (vx.x * vy.y - vx.y * vy.x) * det_1);

	Inv.vt.x=-(Inv.vx.x*vt.x+Inv.vy.x*vt.y+Inv.vz.x*vt.z);
	Inv.vt.y=-(Inv.vx.y*vt.x+Inv.vy.y*vt.y+Inv.vz.y*vt.z);
	Inv.vt.z=-(Inv.vx.z*vt.x+Inv.vy.z*vt.y+Inv.vz.z*vt.z);

	return Inv;
}


//
// Methodes de la classe CRep
//

CRep::CRep(){}

CRep::CRep(const vR3 &v1,const vR3 &v2,const vR3 &v3,const vR3 &v4):mat(v1,v2,v3,v4)
{
	invMat=mat.inv();
}

CRep::CRep(const Transform &m):mat(m),invMat(m.inv()){}

CRep CRep::inv() const
{
	CRep reverse;
	reverse.mat=invMat;
	reverse.invMat=mat;

	return reverse;
}

CRep CRep::operator * (const CRep &cr) const
{
	return CRep(mat*cr.mat);
}


vR3 CRep::Apply(const vR3 &v) const
{
	return mat*v;
}

vR3 CRep::invApply(const vR3 &v) const
{
	return invMat*v;
}

pR3 CRep::Apply(const pR3 &p) const
{
	return mat*p;
}

pR3 CRep::invApply(const pR3 &p) const
{
	return invMat*p;
}

#endif // ALGEBRE_H_
