#ifndef _IMAGE_H_
#define _IMAGE_H_


#include <fstream>
#include <stdlib.h>
#include <string>
#include <list>
#include <cmath>
#include <iostream>
using namespace std;

#include "define.h"
#include "region.h"


/* Point 2 dimension (x, y) permettant la gestion de centre */
/* Rotation, echelle, etc ... */

extern long maxType(size_t type);

class pZ2
{
	public:
		long _x;
		long _y;

		pZ2(long x=0, long y=0) : _x(x), _y(y) {}
		pZ2(const pZ2& pt) : _x(pt._x), _y(pt._y) {}
		~pZ2() {}

		pZ2& operator =(const pZ2& pt)
		{
			_x = pt._x;
			_y = pt._y;
			return *this;
		}
};


template <class T>
class sig1D
{
	friend class Form1;
	friend class ThreadImg1;
	friend class ThreadImg2;
	public:
		T* ptr;
		unsigned ne;

	public: 
		sig1D(unsigned N=0);
		sig1D(const sig1D<T>&);
		~sig1D();

		sig1D<T>& operator =(const sig1D<T>&);

		void set(const T&);
		void Ra0();
		void alloc(unsigned N);

		const T& getMax() const;
		const T& getMin() const;
		int getIndMax() const;	
};




/// Dfinition de la classe signal 1D (histo,...) ///

template <class T>
sig1D<T>::sig1D(unsigned N): ne(N)
{
	ptr = NULL;
	if(N)
		ptr = new T[N];
}

template <class T>
sig1D<T>::sig1D(const sig1D<T>& src) : ne(src.ne)
{
	ptr = NULL;
	if(ne)
	{
		ptr = new T[ne];
		memcpy(ptr, src.ptr, ne*sizeof(T));
	}
}

template <class T>
sig1D<T>::~sig1D<T>()
{
	if(ptr)
		delete[] ptr;
	ptr = NULL;
}

template <class T>
sig1D<T>& sig1D<T>::operator =(const sig1D<T>& src)
{
	if(ptr && src.ne == ne)
		memcpy(ptr, src.ptr, ne*sizeof(T));
	else
	{
		ne = src.ne;

		if(ptr)
		{
			delete[] ptr;
			ptr = NULL;
		}

		if(ne)
		{
			ptr =new T[ne];
			memcpy(ptr, src.ptr, ne*sizeof(T));
		}
	}

	return (*this);
}

template <class T>
void sig1D<T>::set(const T& val)
{
	if(sizeof(T) == 1)
		memset(ptr, val, sizeof(T)*ne);
	else
		for(unsigned i(0);i<ne;i++)
			*(ptr+i) = val;
}

template <class T>
void sig1D<T>::Ra0()
{
	if(sizeof(T) == 1)
		memset(ptr, T(0), sizeof(T)*ne);
	else
		for(int i(0);i<ne;i++)
			*(ptr+i) = T(0);
}

template <class T>
const T& sig1D<T>::getMax() const
{
	if(!ptr)
		throw(0);

	T* maxi = ptr;
	T* cur  = ptr;

	for(int i(ne);i--;cur++)
		if(*maxi < *cur)
			maxi = cur; 

	return *maxi;
}

template <class T>
const T& sig1D<T>::getMin() const
{
	if(!ptr)
		throw(0);

	T* mini = ptr;
	T* cur  = ptr;

	for(int i(ne);i--;cur++)
		if(*mini > *cur)
			mini = cur; 

	return *mini;
}

template <class T>
void sig1D<T>::alloc(unsigned N)
{
	if(ptr)
		delete[] ptr;
	ptr = NULL;

	ne = N;
	ptr = new T[ne];

	set((T)0);
}

template <class T>
int sig1D<T>::getIndMax() const
{
	if(!ptr)
		throw(0);

	T* maxi = ptr;
	T* cur  = ptr;
	int ind(0);

	for(unsigned i(0);i<ne;i++, cur++)
		if(*maxi < *cur)
		{
			ind = i;
			maxi = cur;
		}

	return ind;
}


/////////////////////////////////////////////////////

class EStruct;

template <class T>
class Image
{

	friend class ImgCouleur;
	friend class Form1;
	friend class ThreadImg1;
	friend class ThreadImg2;
private:
		inline void PPV(int& _x, int& _y, const float& _xx, const float& _yy) const
		{
			_x = _xx + 0.5;
			_y = _yy + 0.5;
		}

		int  IBL(const float&, const float&) const;

	protected:
		int		width;
		int		height;
		T*		_data;

	public:
		Image(unsigned _w=0, unsigned _h=0);
		Image(const Image<T>&);
		~Image();

		Image<T>& operator =(const Image<T>&);

		void load(const char*, const char* _type="PGM");
		void save(const char*, const char* _type="PGM");
		inline void set(const T& _val)
		{
			T* ptrm = _data;
			for(int i(width*height); i--;)
				*(ptrm++) = _val;
		}

		void alloc(unsigned, unsigned);
		sig1D<ULONG> histogramme() const;

		/* Operations sur images */

		Image<T> operator +(const Image<T>&) const;
		Image<T> operator -(const Image<T>&) const;
		Image<T> operator &(const Image<T>&) const;

		/* Erosion / Dilatation, Ouverture / Fermeture */

		Image<T> dilatation(const EStruct& elem) const;
		Image<T> erosion(const EStruct& elem) const;
		inline Image<T> ouverture(const EStruct& elem) const
		{
			Image<T> res(width, height);
			res.set((UCHAR)BLANC);
			res = this->erosion(elem).dilatation(elem);
			return res;
		}

		inline Image<T> fermeture(const EStruct& elem) const
		{
			Image<T> res(width, height);
			res.set((UCHAR)BLANC);
			res = this->dilatation(elem).erosion(elem);
			return res;
		}




		/* Transformations gomtriques */

		Image<T> rotation(const float&, const pZ2&) const;
		Image<T> translation(int, int) const;
		Image<T> echele(const float&, const float&, const pZ2&) const;
		Image<T> cisaille(const float&, const float&) const;
		Image<T> flip() const;
		Image<T>& mirror() const;
		Image<T> anamorphose() const;
		Image<T> egalHisto() const;




		/* 'Utilitaires' utiles */

		Image<T> traceHisto(unsigned nbc=256, unsigned nbl=256) const;
		Image<T> seuillageMin(const T&) const;
		Image<T> seuillageMax(const T&) const;
		Image<T> chapeauHaut(const EStruct& elem, unsigned type=BLANC) const;
		Image<T> LPE() const;
		void traceBordure(int);
		T getSeuilMin() const;

		inline T* getBuffer()
		{	return _data;	}
			
		inline int getWidth() const
		{	return width;	}

		inline int getHeight() const
		{	return height;	}


};


class EStruct
{
	friend class Image<class T>;
	friend class Image<UCHAR>;
	friend class Image<ULONG>;
	
	protected:
		int* mat;
		unsigned dim;
		double coef;

	public:
		EStruct(unsigned d=0, const double& c=1.);
		EStruct(const EStruct&);
		~EStruct();

		EStruct& operator =(const EStruct&);

		void set(int);
		void setDim(unsigned);
		inline void EStruct::setCoef(const double& c)
		{	coef = c;	}


		int* operator[](unsigned);
		int* operator[](unsigned) const;

		template <class T> int operator ()(int, int, const Image<T>&) const;

		inline void EStruct::affiche() const
		{
			for(unsigned i(0);i<dim;i++)
				for(unsigned j(0);j<dim;j++)
					cout << *(mat + i*dim + j) << " ";
				cout << endl;
		}

};


template <class T>
Image<T>::Image(unsigned _w, unsigned _h) : width(_w), height(_h)
{
	_data = NULL;
	if(_w*_h)
		_data = new T[width*height];
}

template <class T>
Image<T>::Image(const Image<T>& _src) : width(_src.width), height(_src.height)
{
	_data = NULL;
	if(_src.height*_src.width)
	{
		_data = new T[width*height];
		memcpy(_data, _src._data, sizeof(T)*width*height);
	}
}

template <class T>
Image<T>::~Image<T>()
{
	if(_data)
		delete[] _data;
	_data = NULL;
}

template <class T>
Image<T>& Image<T>::operator =(const Image<T>& _src)
{
	if(_data)
		delete[] _data;
	_data = NULL;

	width  = _src.width;
	height = _src.height;

	if(_src.height*_src.width)
	{
		_data = new T[width*height];
		memcpy(_data, _src._data, sizeof(T)*width*height);
	}

	return (*this);
}

template <class T>
void Image<T>::load(const char* fic, const char* _type)
{
	if(string(_type) == "PGM")
	{
		ifstream flit;
		flit.open(fic, ios_base::binary);

		string format;
		string joe;
		char antn,tmp[128];

		flit >> format;
		/*
		while(!flit.eof() )
		{
			flit.getline(tmp,sizeof(char)*128);
			if(tmp[0]!='#')
			break;
		}*/
		
		flit >> width;
		//width=atoi(tmp);
		flit >> height;
		
		flit >> joe;
		
		if(_data)
			delete [] _data;	
		//if(!_data)
			_data = new T[width*height];

		flit.read(&antn, sizeof(char));
		flit.read((char*)_data, width*height*sizeof(T));

		flit.close();
	}
}

template <class T>
void Image<T>::save(const char* fic, const char* _type)
{
	if(string(_type) == "PGM")
	{
		ofstream flot(fic, ios_base::binary);

		flot << "P5\n" << width << " " << height << "\n255\n";
    	flot.write((char*)_data, height*width*sizeof(T));

		flot.close();
	}
	
}


template <class T>
void Image<T>::alloc(unsigned w, unsigned h)
{
	if(_data)
		delete[] _data;

	_data = new T[w*h];

	width  = w;
	height = h;
}

template <class T>
sig1D<ULONG> Image<T>::histogramme() const
{
	T* ptrm = _data;
	ULONG* pt_histo = NULL;
	sig1D<ULONG> histo((int)pow(2.0, (int)sizeof(T)*8));
	histo.set(T(0));
	pt_histo = histo.ptr;

	for(int i(width*height); i--;)
		++(pt_histo[*(ptrm++)]);

	return histo;
}

template <class T>
Image<T> Image<T>::traceHisto(unsigned nbc, unsigned nbl) const
{
	sig1D<ULONG>	histo(histogramme());
	ULONG*			pt_histo = histo.ptr;
	T*				ptrm = _data;

	long maxi(histo.getMax());
	Image<UCHAR> iHist(nbc, nbl);
	UCHAR* ptr = iHist._data;
	double rapport;

	rapport = (double)255 / (double)maxi;

	iHist.set(NOIR);

	for(int k(0);k<nbc;k++,pt_histo++)
		for(int j(0);j<(int)(*pt_histo * rapport);j++)
			*(ptr + (255-j)*nbc+k) = (UCHAR)BLANC;

	return iHist;
}

template <class T>
Image<T> Image<T>::operator +(const Image<T>& src) const
{
	if(width != src.width || height != src.height)
		return Image<T> (src);

	Image<T> res(src.width, src.height);
	T* pt1, *pt2, *ptd;
	pt1 = _data;
	pt2 = src._data;
	ptd = res._data;

	for(int i(0);i<width*height;i++,pt1++,pt2++,ptd++)
		*(ptd) = min(static_cast<int>(*pt1)+static_cast<int>(*pt2), 255);

	return res;
}

template <class T>
Image<T> Image<T>::operator -(const Image<T>& src) const
{
	if(width != src.width || height != src.height)
		return Image<T> (src);

	Image<T> res(src.width, src.height);
	T* pt1, *pt2, *ptd;
	pt1 = _data;
	pt2 = src._data;
	ptd = res._data;

	for(int i(0);i<width*height;i++,pt1++,pt2++,ptd++)
		*(ptd) = max(static_cast<int>(*pt1)-static_cast<int>(*pt2), 0);

	return res;
}

template <class T>
Image<T> Image<T>::operator &(const Image<T>& src) const
{
	if(width != src.width || height != src.height)
		return Image<T> (src);

	Image<T> res(src.width, src.height);
	T* pt1, *pt2, *ptd;
	pt1 = _data;
	pt2 = src._data;
	ptd = res._data;

	for(int i(0);i<width*height;i++,pt1++,pt2++,ptd++)
		*(ptd) = (*pt1)&(*pt2);

	return res;
}

template <class T>
Image<T> Image<T>::rotation(const float& angle, const pZ2& C) const
{
	int xp;
	int yp;
	Image<T> res(width, height);
	T* ptm = res._data;

	res.set((UCHAR)BLANC);

	for(int i(0);i<height;i++)
		for(int j(0);j<width;j++)
		{
			/* Mthode du plus propche voisin (interpolation ordre 0) */

			//PPV(xp, yp, (double)C._x + ((double)j-(double)C._x)*cos(angle) - ((double)i-(double)C._y)*sin(angle), (double)C._y + ((double)i-(double)C._y)*cos(angle) + ((double)j-(double)C._x)*sin(angle));
			
			int val = IBL((double)C._x + ((double)j-(double)C._x)*cos(angle) -
				((double)i-(double)C._y)*sin(angle), (double)C._y + 
				((double)i-(double)C._y)*cos(angle) + ((double)j-(double)C._x)*sin(angle));

			if(val > -1)
				*(ptm + i * width+j) = val;

			/*if(xp < width && xp >= 0 && yp < height && yp >= 0)
				*(ptm + i*width+j) = *(_data + xp*width + yp);*/
		}
	
	return res;
}

template <class T>
Image<T> Image<T>::translation(int dx, int dy) const
{
	Image<T> res(width, height);
	T* ptm = res._data;

	res.set((UCHAR)BLANC);

	for(int i(0);i<height;i++)
		for(int j(0);j<width;j++)
			if(i-dy > 0 && j-dx > 0)
				*(ptm + i*width+j) = *(_data + (i-dy)*width+j-dy);

	return res;
}

template <class T>
Image<T> Image<T>::echele(const float& dx, const float& dy, const pZ2& C) const
{
	int val;
	Image<T> res(width, height);

	res.set((UCHAR)BLANC);

	for(int i(0);i<height;i++)
		for(int j(0);j<width;j++)
		{
			val = IBL((float)C._x + (float)(j-C._x) * dx, (float)C._y+ (float)(i-C._y) * dy);

			if(val > 0)
				*(res._data + i*width + j) = val;
		}

	return res;
}

template <class T>
Image<T> Image<T>::cisaille(const float& dx, const float& dy) const
{
	Image<T> res(width, height);

	res.set((UCHAR)BLANC);

	for(int i(0);i<height;i++)
		for(int j(0);j<width;j++)
		{
			int xp(dx*i+j);
			int yp(dy*j+i);

			if(xp < height && yp < width)
				*(res._data + i*width + j) = *(_data + yp * width + xp);
		}

	return res;
}

template <class T>
Image<T> Image<T>::flip() const
{
	Image<T> res(width, height);

	res.set((UCHAR)BLANC);

	for(int i(0);i<height;i++)
		for(int j(0);j<width;j++)
				*(res._data + i*width + j) = *(_data + i * width + width - 1 - j);

	return res;
}

template <class T>
Image<T>& Image<T>::mirror() const
{/*
	Image<T> res(width, height);

	res.set((UCHAR)BLANC);

	for(int i(0);i<height;i++)
		for(int j(0);j<width;j++)
				*(res._data + i*width + j) = *(_data + (height-1-i) * width + j);

	return res;*/
	Image<T> *res=new Image<T>(width, height);

	res->set((UCHAR)BLANC);

	for(int i(0);i<height;i++)
		for(int j(0);j<width;j++)
				*(res->_data + i*width + j) = *(_data + (height-1-i) * width + j);

	return *res;
}

template <class T>
int Image<T>::IBL(const float& _yp, const float& _xp) const
{
	if(_xp < height-1 && _yp < width-1 && _xp >= 0. && _yp >= 0.)
	{
		int x0((int)_xp), y0((int)_yp), p00(x0*width+y0),
			p01(x0*width+y0+1), p10((x0+1)*width+y0), p11((x0+1)*width+y0+1);
		float ex(_xp-x0 ), ey( _yp-y0 );

		float res( (1-ex)*(1-ey)*((int)*(_data+p00))+
			 (1-ex)*ey*((int)*(_data+p01))+ex*(1-ey)*((int)*(_data+p10))+ex*ey*((int)*(_data+p11)));

		return res;
	}
	return -1;

}

template <class T>
Image<T> Image<T>::anamorphose() const
{
	long* histo = new long[(int)pow(2.0, (int)sizeof(T)*8)];
	long* pt_histo = histo;
	T* ptrm = _data;
	Image<T> res(width, height);
	T a, b;

	memset(histo, 0, sizeof(int)*256);

	for(int i(width*height); i--;)
		++(pt_histo[*(ptrm++)]);

	for(int j(0);j<256;j++)
		if(histo[j] != 0)
		{
			a = j;
			break;
		}

	for(int k(255);k>=0;k--)
		if(histo[k] != 0)
		{
			b = k;
			break;
		}

	T* LUT = new T[256];
	memset(LUT, 0, sizeof(T)*256);

	for(int l(0);l<256;l++)
		LUT[l] = (T) ( 255.0 * fabs((float)(l - a)) / ((float)(b - a)));

	for(int ii(0);ii<height;ii++)
		for(int jj(0);jj<width;jj++)
			*(res._data + ii*width + jj) = LUT[*(_data + ii*width+jj)];

	if(histo)
		delete[] histo;
	if(LUT)
		delete[] LUT;

	return res;
}

template <class T>
Image<T> Image<T>::egalHisto() const
{
	sig1D<ULONG> histo(histogramme());
	sig1D<double> histC(histo.ne);
	Image<T> res(width, height);
	double product((double)(width*height));
	double* pt_hist = histC.ptr;
	sig1D<double> probC(histo.ne);
	double* pt_prob = probC.ptr;
	ULONG* pt_h = histo.ptr;
	T* pt_d = res._data;
	
	for(unsigned i(histo.ne);i--;)
		*(pt_hist++) = ((double)*(pt_h++)) / product;

	probC.set(T(0));
	
	for(unsigned j(0);j<histo.ne;j++, pt_prob++)
		for(unsigned k(0);k<j;k++)
			*pt_prob += *(histC.ptr + k);

	pt_prob = probC.ptr;

	for(unsigned i(0);i<width*height;i++)
		*(pt_d++) = (T)(*(pt_prob+*(_data+i))*255.);

	return res;
}

template <class T>
Image<T> Image<T>::seuillageMin(const T& vMin) const
{
	Image<T> res(width, height);
	T* pt_d = res._data;
	T* pt_s = _data;
	res.set((T)NOIR);

	for(int i(width*height);i--;pt_d++, pt_s++)
		if(*pt_s > vMin)
				*pt_d = T(maxType(sizeof(T)));

	return res;
}

template <class T>
Image<T> Image<T>::seuillageMax(const T& vMax) const
{
	Image<T> res(width, height);
	T* pt_d = res._data;
	T* pt_s = _data;
	res.set((T)NOIR);

	for(int i(width*height);i--;pt_d++, pt_s++)
		if(*pt_s < vMax)
				*pt_d = T(maxType(sizeof(T)));

	return res;
}


template <class T>
Image<T> Image<T>::dilatation(const EStruct& elem) const
{
	Image<T> res(width, height);

	res.set((UCHAR)BLANC);

	T minim;

	for(int i(((int)elem.dim)/2);i<((int)height)-((int)elem.dim)/2;i++)
		for(int j(((int)elem.dim)/2);j<((int)width)-((int)elem.dim)/2;j++)
		{
			int a(0), b;
			minim = maxType(sizeof(T));

			for(int k(-((int)elem.dim)/2);k<=((int)elem.dim)/2;k++, a++)
			{
				b = 0;
				
				for(int l(-((int)elem.dim)/2);l<=((int)elem.dim)/2;l++, b++)
					if(elem[a][b] == 1)
						minim = min(*(_data + (i+k) * width + j+l), minim);
			}

			*(res._data+i*width+j) = minim;
		}
	return res;
}

template <class T>
Image<T> Image<T>::erosion(const EStruct& elem) const
{
	Image<T> res(width, height);

	res.set((UCHAR)BLANC);

	T maxim;

	for(int i(((int)elem.dim)/2);i<height-((int)elem.dim)/2;i++)
		for(int j(((int)elem.dim)/2);j<width-((int)elem.dim)/2;j++)
		{
			int a(0), b;
			maxim = (T)0;

			for(int k(-((int)elem.dim)/2);k<=((int)elem.dim)/2;k++, a++)
			{
				b = 0;
				
				for(int l(-((int)elem.dim)/2);l<=((int)elem.dim)/2;l++, b++)
					if(elem[a][b] == 1)
						maxim = max(*(_data + (i+k) * width + j+l), maxim);
			}

			*(res._data+i*width+j) = maxim;
		}
	return res;
}





template <class T>
Image<T> Image<T>::chapeauHaut(const EStruct& elem, unsigned type) const
{
	int tmp;
	Image<T> res(width, height);
	Image<T> dilat;
	
	if(type == BLANC)
		dilat = ouverture(elem);
	else
		dilat = fermeture(elem);

	res.set(type);

	for(int i(0);i<width*height;i++)
	{
		tmp = (type == BLANC ? ((int)*(dilat._data + i)) - ((int)*(_data + i)) : ((int)*(_data + i)) - ((int)*(dilat._data + i)) );

		if(tmp < 0)
			tmp = 0;

		*(res._data + i) = (UCHAR)tmp;
	}

	return res;
}


template <class T>
Image<T> Image<T>::LPE() const
{
	Image<T> res(width, height);
	res.set(NOIR);

/*#ifndef INIT
	#define INIT -1;
#endif
#ifndef MASK
	#define MASK -2;
#endif
#ifndef WSHED
	#define WSHED 0;
#endif
#ifndef IQUEUE
	#define IQUEUE -3;
#endif

	
	sig1D<ULONG> histo(histogramme());
	sig1D<ULONG> hcum(histo.ne);
	int* ImgLab = new int[height*width];
	int Ims[height*width];
	list<int> Fifo;

	for(int i(0);i<width*height;i++)
		*(ImgLab+i) = INIT;
	
	*(hcum.ptr) = *(histo.ptr);

	for(int i(1);i<histo.ne;i++)
		*(hcum.ptr + i) = *(histo.ptr + i) + *(hcum.ptr + i - 1);
	
	for(int i(0);i<width*height;i++)
		*(Ims + *(hcum.ptr+*(_data+i))) = i; 

	UCHAR pixel;
	int p, pp;

	for(int i(width);i<((int)width*height)-((int)width);i++)
	{
		pixel = *(_data + Ims[i]);

		ImgLab[Ims[i]] = MASK;

		if(Ims[i-1] >= WSHED || Ims[i+1] >= WSHED || Ims[i-width] >= WSHED || Ims[i+width] >= WSHED)
		{
			ImgLab[Ims[i]] = IQUEUE;
			Fifo.push_back(Ims[i]);
		}

		while(Fifo.size())
		{
			p = Fifo.front();
			Fifo.pop_front();

			pp = ImgLab[p]
		}
	}

*/
	return res;
}

template <class T>
void Image<T>::traceBordure(int dim)
{
	for(int i(0);i<dim/2;i++)
		for(int j(0);j<width;j++)
			*(_data+i*width+j) = 0;

	for(int i(((int)height)-dim/2);i<height;i++)
		for(int j(0);j<width;j++)
			*(_data+i*width+j) = 0;

	for(int i(0);i<height;i++)
		for(int j(0);j<dim/2;j++)
			*(_data+i*width+j) = 0;

	for(int i(0);i<height;i++)
		for(int j(width-dim/2);j<width;j++)
			*(_data+i*width+j) = 0;
}


template <class T>
T Image<T>::getSeuilMin() const
{
	return histogramme().getIndMax();
}

///////////////////////////// IMAGE COULEUR ///////////////////////////////////////////


class ImgCouleur : public Image<ULONG>
{
	public:
		ImgCouleur(unsigned h=0, unsigned w=0);
		ImgCouleur(const ImgCouleur&);
		~ImgCouleur();

		ImgCouleur& operator =(const ImgCouleur&);

		bool savePPM(const char*) const;
		bool loadPPM(const char*);

		sig1D<ULONG>* histogramme() const;

		ImgCouleur* traceHisto(unsigned nbc, unsigned nbl) const;
		
		inline Image<UCHAR> ImgCouleur::getLum() const
		{
			Image<UCHAR> res(width, height);
			for(int i(0);i<width*height;i++)
				*(res._data + i) = static_cast<UCHAR>((((int)getRed(i) + (int)getGreen(i) + (int)getBlue(i)) / 3));
			return res;
		}
		
		inline UCHAR ImgCouleur::getRed(unsigned i) const
		{
			return (*(_data + i)) >> 16;
		}
		
		inline UCHAR ImgCouleur::getGreen(unsigned i) const
		{
			return ((*(_data + i)) >> 8) & 0x000000ff;
		}
		
		inline UCHAR ImgCouleur::getBlue(unsigned i) const
		{
			return (*(_data + i)) & 0x000000ff;
		}
		static ImgCouleur Colorisation(Image<UCHAR>& src, Stat&);
};

class ImgNdg : public Image<UCHAR>
{};


#endif // _IMAGE_H_
